#!/bin/bash
# WordChat Plugin WP-CLI Test Script
#
# Tests plugin functionality using WP-CLI commands
# Requires: WP-CLI installed and WordPress site set up
#
# Usage:
#   ./bin/test-plugin.sh
#   ./bin/test-plugin.sh --site=http://localhost:8080

set -e

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m'

echo -e "${YELLOW}==================================${NC}"
echo -e "${YELLOW}WordChat Plugin WP-CLI Test Suite${NC}"
echo -e "${YELLOW}==================================${NC}"
echo ""

# Parse arguments
SITE_URL=""
if [ "$1" != "" ]; then
    if [[ $1 == --site=* ]]; then
        SITE_URL="${1#*=}"
    fi
fi

# Check if WP-CLI is installed
if ! command -v wp &> /dev/null; then
    echo -e "${RED}Error: WP-CLI not installed${NC}"
    echo "Install from: https://wp-cli.org/"
    exit 1
fi

echo -e "${GREEN}✓${NC} WP-CLI found"

# Set WP-CLI command with site URL if provided
WP_CMD="wp"
if [ "$SITE_URL" != "" ]; then
    WP_CMD="wp --url=$SITE_URL"
fi

# Test 1: Check if plugin exists
echo ""
echo "Test 1: Check plugin exists..."
if $WP_CMD plugin list --name=wordchat --format=count | grep -q "1"; then
    echo -e "${GREEN}✓${NC} Plugin found"
else
    echo -e "${RED}✗${NC} Plugin not found"
    exit 1
fi

# Test 2: Activate plugin
echo ""
echo "Test 2: Activate plugin..."
if $WP_CMD plugin activate wordchat 2>/dev/null; then
    echo -e "${GREEN}✓${NC} Plugin activated"
else
    echo -e "${YELLOW}⚠${NC}  Plugin already active or error"
fi

# Test 3: Check if plugin is active
echo ""
echo "Test 3: Verify plugin is active..."
if $WP_CMD plugin list --status=active --name=wordchat --format=count | grep -q "1"; then
    echo -e "${GREEN}✓${NC} Plugin is active"
else
    echo -e "${RED}✗${NC} Plugin is not active"
    exit 1
fi

# Test 4: Check options exist
echo ""
echo "Test 4: Check WordChat options..."

SITE_ID=$($WP_CMD option get wordchat_site_id 2>/dev/null || echo "")
API_KEY=$($WP_CMD option get wordchat_api_key 2>/dev/null || echo "")

if [ "$SITE_ID" != "" ] && [ "$API_KEY" != "" ]; then
    echo -e "${GREEN}✓${NC} Site registered (site_id and api_key exist)"
    echo "  Site ID: ${SITE_ID:0:10}..."
    echo "  API Key: ${API_KEY:0:10}..."
else
    echo -e "${YELLOW}⚠${NC}  Site not yet registered (run activation)"
fi

# Test 5: Check license status
echo ""
echo "Test 5: Check license status..."

LICENSE_KEY=$($WP_CMD option get wordchat_license_key 2>/dev/null || echo "")
IS_PAID=$($WP_CMD option get wordchat_is_paid 2>/dev/null || echo "0")

if [ "$LICENSE_KEY" != "" ]; then
    echo -e "${GREEN}✓${NC} License key exists: ${LICENSE_KEY:0:15}..."
else
    echo -e "${YELLOW}⚠${NC}  No license key (free plan)"
fi

if [ "$IS_PAID" == "1" ]; then
    echo -e "${GREEN}✓${NC} Pro plan active"
else
    echo -e "${YELLOW}⚠${NC}  Free plan"
fi

# Test 6: Create test post
echo ""
echo "Test 6: Create test post..."

TEST_POST_ID=$($WP_CMD post create \
    --post_title="WordChat Test Post $(date +%s)" \
    --post_content="This is a test post for WordChat plugin testing." \
    --post_status=publish \
    --post_type=post \
    --porcelain 2>/dev/null)

if [ "$TEST_POST_ID" != "" ]; then
    echo -e "${GREEN}✓${NC} Test post created (ID: $TEST_POST_ID)"
else
    echo -e "${RED}✗${NC} Failed to create test post"
fi

# Test 7: Check post meta
echo ""
echo "Test 7: Check sync metadata..."

SYNCED=$($WP_CMD post meta get $TEST_POST_ID wordchat_synced 2>/dev/null || echo "")
SYNC_STATUS=$($WP_CMD post meta get $TEST_POST_ID wordchat_sync_status 2>/dev/null || echo "")

echo "  Synced: ${SYNCED:-'not set'}"
echo "  Sync Status: ${SYNC_STATUS:-'not set'}"

# Test 8: List all posts with sync status
echo ""
echo "Test 8: List synced content..."

echo "Posts synced to WordChat:"
$WP_CMD post list \
    --post_type=post,page \
    --post_status=publish \
    --meta_key=wordchat_synced \
    --meta_value=1 \
    --fields=ID,post_title,post_date \
    --format=table 2>/dev/null || echo "  None"

# Test 9: Check cron jobs
echo ""
echo "Test 9: Check scheduled cron jobs..."

if $WP_CMD cron event list --format=count 2>/dev/null | grep -q "[0-9]"; then
    echo "WordChat cron jobs:"
    $WP_CMD cron event list | grep wordchat || echo "  None scheduled"
fi

# Test 10: Test plugin deactivation
echo ""
echo "Test 10: Test deactivation..."

if $WP_CMD plugin deactivate wordchat 2>/dev/null; then
    echo -e "${GREEN}✓${NC} Plugin deactivated"

    # Check cron cleanup
    CRON_COUNT=$($WP_CMD cron event list | grep wordchat | wc -l || echo "0")
    if [ "$CRON_COUNT" == "0" ]; then
        echo -e "${GREEN}✓${NC} Cron jobs cleaned up"
    else
        echo -e "${YELLOW}⚠${NC}  Cron jobs still exist"
    fi
fi

# Reactivate
if $WP_CMD plugin activate wordchat 2>/dev/null; then
    echo -e "${GREEN}✓${NC} Plugin reactivated"
fi

# Test 11: Check for JavaScript/CSS assets
echo ""
echo "Test 11: Check plugin assets..."

PLUGIN_DIR=$($WP_CMD plugin path wordchat 2>/dev/null)

if [ -f "$PLUGIN_DIR/js/chat.js" ]; then
    echo -e "${GREEN}✓${NC} chat.js exists"
else
    echo -e "${RED}✗${NC} chat.js missing"
fi

if [ -f "$PLUGIN_DIR/css/admin.css" ]; then
    echo -e "${GREEN}✓${NC} admin.css exists"
else
    echo -e "${YELLOW}⚠${NC}  admin.css missing (may not exist)"
fi

# Test 12: Verify plugin version
echo ""
echo "Test 12: Check plugin version..."

VERSION=$($WP_CMD plugin list --name=wordchat --field=version 2>/dev/null)
echo "  Current version: $VERSION"

# Cleanup test post
echo ""
echo "Cleanup: Deleting test post..."
if [ "$TEST_POST_ID" != "" ]; then
    $WP_CMD post delete $TEST_POST_ID --force 2>/dev/null
    echo -e "${GREEN}✓${NC} Test post deleted"
fi

# Summary
echo ""
echo -e "${YELLOW}==================================${NC}"
echo -e "${YELLOW}Test Summary${NC}"
echo -e "${YELLOW}==================================${NC}"
echo -e "${GREEN}✓${NC} Plugin functional tests passed"
echo ""
echo "Plugin Status:"
echo "  Active: Yes"
echo "  Version: $VERSION"
echo "  Registered: ${SITE_ID:+Yes}${SITE_ID:-No}"
echo "  License: ${LICENSE_KEY:+Pro}${LICENSE_KEY:-Free}"
echo ""
echo -e "${GREEN}All tests completed successfully!${NC}"
