<?php
// includes/api-proxy.php
// Server-side API proxy to protect credentials from client exposure

// AJAX handler for chat queries (available to all users, logged in or not)
add_action('wp_ajax_wordchat_query', 'wordchat_handle_query');
add_action('wp_ajax_nopriv_wordchat_query', 'wordchat_handle_query');

/**
 * Handle chat query requests via WordPress AJAX
 * This keeps API credentials on the server and never exposes them to the browser
 */
function wordchat_handle_query() {
    // Verify nonce for CSRF protection
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wordchat_query_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed'], 403);
        wp_die();
    }

    // Rate limiting: Check request count per IP
    if (!wordchat_check_rate_limit()) {
        wp_send_json_error(['message' => 'Too many requests. Please wait a moment and try again.'], 429);
        wp_die();
    }

    // Get credentials from server (NOT exposed to client)
    $site_id = get_option('wordchat_site_id');
    $api_key = get_option('wordchat_api_key');

    // Validate credentials exist
    if (empty($site_id) || empty($api_key)) {
        wp_send_json_error(['message' => 'Plugin not configured. Please contact the site administrator.'], 500);
        wp_die();
    }

    // Get and sanitize question from POST
    $question = isset($_POST['question']) ? sanitize_text_field($_POST['question']) : '';

    if (empty($question)) {
        wp_send_json_error(['message' => 'Question is required'], 400);
        wp_die();
    }

    // Validate question length
    if (strlen($question) > 500) {
        wp_send_json_error(['message' => 'Question is too long (max 500 characters)'], 400);
        wp_die();
    }

    // Make server-to-server API request to backend
    $response = wp_remote_post(WORDCHAT_API_URL . '/query', [
        'headers' => ['Content-Type' => 'application/json'],
        'body' => json_encode([
            'site_id' => $site_id,
            'api_key' => $api_key,
            'domain' => get_site_url(),
            'question' => $question
        ]),
        'timeout' => 60,
        'blocking' => true
    ]);

    // Handle WordPress HTTP errors
    if (is_wp_error($response)) {
        error_log('WordChat API Error: ' . $response->get_error_message());
        wp_send_json_error(['message' => 'Unable to connect to chat service. Please try again.'], 500);
        wp_die();
    }

    // Get response details
    $status_code = wp_remote_retrieve_response_code($response);
    $content_type = wp_remote_retrieve_header($response, 'content-type');

    // Handle streaming response (SSE)
    if (strpos($content_type, 'text/event-stream') !== false) {
        // For streaming responses, we need to forward the stream
        // Set appropriate headers
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no');

        // Stream the response body
        echo wp_remote_retrieve_body($response);
        wp_die();
    }

    // Handle JSON response
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    // Forward the response to the client
    if ($status_code === 200) {
        wp_send_json_success($data, 200);
    } else {
        // Extract error message
        $error_message = isset($data['error']) ? $data['error'] : 'An error occurred';
        wp_send_json_error(['message' => $error_message], $status_code);
    }

    wp_die();
}

/**
 * Rate limiting: Allow max 20 requests per minute per IP
 * Uses WordPress transients for storage
 */
function wordchat_check_rate_limit() {
    // Get user IP address
    $ip = wordchat_get_client_ip();

    // Create transient key based on IP
    $transient_key = 'wordchat_rate_' . md5($ip);

    // Get current count
    $count = get_transient($transient_key);

    // If no transient exists, create it
    if ($count === false) {
        set_transient($transient_key, 1, 60); // 1 minute window
        return true;
    }

    // Check if limit exceeded (20 requests per minute)
    if ($count >= 20) {
        error_log("WordChat: Rate limit exceeded for IP: $ip");
        return false;
    }

    // Increment count
    set_transient($transient_key, $count + 1, 60);
    return true;
}

/**
 * Get client IP address (handles proxies and load balancers)
 */
function wordchat_get_client_ip() {
    $ip_keys = [
        'HTTP_CF_CONNECTING_IP', // Cloudflare
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_REAL_IP',
        'REMOTE_ADDR'
    ];

    foreach ($ip_keys as $key) {
        if (isset($_SERVER[$key]) && !empty($_SERVER[$key])) {
            $ip = $_SERVER[$key];
            // Handle comma-separated IPs (proxies)
            if (strpos($ip, ',') !== false) {
                $ips = explode(',', $ip);
                $ip = trim($ips[0]);
            }
            // Validate IP format
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }

    return '0.0.0.0'; // Fallback
}

// AJAX handler for logging chat interactions
add_action('wp_ajax_wordchat_log_chat', 'wordchat_handle_log_chat');
add_action('wp_ajax_nopriv_wordchat_log_chat', 'wordchat_handle_log_chat');

/**
 * Handle chat logging requests via WordPress AJAX
 */
function wordchat_handle_log_chat() {
    // Verify nonce for CSRF protection
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wordchat_query_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed'], 403);
        wp_die();
    }

    // Get credentials from server
    $site_id = get_option('wordchat_site_id');
    $api_key = get_option('wordchat_api_key');

    if (empty($site_id) || empty($api_key)) {
        wp_send_json_error(['message' => 'Plugin not configured'], 500);
        wp_die();
    }

    // Get POST data
    $question = isset($_POST['question']) ? sanitize_text_field($_POST['question']) : '';
    $answer = isset($_POST['answer']) ? sanitize_textarea_field($_POST['answer']) : '';
    $sources = isset($_POST['sources']) ? json_decode(stripslashes($_POST['sources']), true) : [];
    $navigated_to_url = isset($_POST['navigated_to_url']) ? esc_url_raw($_POST['navigated_to_url']) : null;

    // Make server-to-server API request
    $response = wp_remote_post(WORDCHAT_API_URL . '/log-chat', [
        'headers' => ['Content-Type' => 'application/json'],
        'body' => json_encode([
            'site_id' => $site_id,
            'api_key' => $api_key,
            'domain' => get_site_url(),
            'question' => $question,
            'answer' => $answer,
            'sources' => $sources,
            'navigated_to_url' => $navigated_to_url
        ]),
        'timeout' => 15,
        'blocking' => false // Non-blocking - fire and forget
    ]);

    // Return success even if logging fails (non-critical)
    wp_send_json_success(['logged' => true], 200);
    wp_die();
}
