<?php
// includes/sync.php

function wordchat_render_sync_page() {
    $site_id = get_option('wordchat_site_id');
    $api_key = get_option('wordchat_api_key');

    if (!$site_id || !$api_key) {
        echo "<div class='notice notice-error'><p>" . esc_html__('Please register Word.Chat before syncing content.', 'wordchat') . "</p></div>";
        return;
    }

    $args = ['post_type' => ['post', 'page'], 'post_status' => 'publish', 'numberposts' => -1];
    $posts = get_posts($args);

    $synced_ids = [];
    foreach ($posts as $post) {
        if (get_post_meta($post->ID, 'wordchat_synced', true)) {
            $synced_ids[] = $post->ID;
        }
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Verify nonce
        if (!isset($_POST['wordchat_sync_nonce']) || !wp_verify_nonce($_POST['wordchat_sync_nonce'], 'wordchat_sync_action')) {
            wp_die(__('Security check failed. Please try again.', 'wordchat'));
        }

        $selected_ids = isset($_POST['wordchat_sync']) ? array_map('absint', $_POST['wordchat_sync']) : [];
        $unsynced_ids = array_diff($synced_ids, $selected_ids);

        // Sync selected
        foreach ($selected_ids as $post_id) {
            $post = get_post($post_id);
            if (!$post) continue;

            $url = get_permalink($post);
            $categories = get_the_category($post_id);
            $category = $categories && !empty($categories) ? $categories[0]->name : null;

            $payload = [
                'site_id' => $site_id,
                'api_key' => $api_key,
                'domain' => get_site_url(),
                'chunks' => [[
                    'text' => strip_tags($post->post_title . ". " . $post->post_content),
                    'url' => $url,
                    'category' => $category
                ]]
            ];

            $response = wp_remote_post(WORDCHAT_API_URL . '/ingest', [
                'headers' => ['Content-Type' => 'application/json'],
                'body' => json_encode($payload)
            ]);

            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                update_post_meta($post_id, 'wordchat_synced', true);
                update_post_meta($post_id, 'wordchat_synced_at', current_time('mysql'));
            }
        }

        // Unsync deselected
        foreach ($unsynced_ids as $post_id) {
            $post = get_post($post_id);
            if (!$post) continue;

            $url = get_permalink($post);

            $payload = [
                'site_id' => $site_id,
                'api_key' => $api_key,
                'url' => $url
            ];

            $response = wp_remote_post(WORDCHAT_API_URL . '/unsync', [
                'headers' => ['Content-Type' => 'application/json'],
                'body' => json_encode($payload)
            ]);

            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                delete_post_meta($post_id, 'wordchat_synced');
                delete_post_meta($post_id, 'wordchat_synced_at');
            }
        }

        echo "<div class='notice notice-success'><p>" . esc_html__('Sync updated successfully.', 'wordchat') . "</p></div>";
    }

    echo '<div class="wrap"><h2>' . esc_html__('Sync WordPress Content to Word.Chat', 'wordchat') . '</h2>';
    echo '<form method="post">';
    wp_nonce_field('wordchat_sync_action', 'wordchat_sync_nonce');
    echo '<table class="widefat"><thead><tr>';
    echo '<th style="width:30px;"></th>';
    echo '<th>' . esc_html__('Title', 'wordchat') . '</th>';
    echo '<th>' . esc_html__('Type', 'wordchat') . '</th>';
    echo '<th>' . esc_html__('Published', 'wordchat') . '</th>';
    echo '<th>' . esc_html__('Modified', 'wordchat') . '</th>';
    echo '<th>' . esc_html__('Status', 'wordchat') . '</th>';
    echo '<th>' . esc_html__('Last Synced', 'wordchat') . '</th>';
    echo '</tr></thead><tbody>';

    foreach ($posts as $post) {
        $is_synced = get_post_meta($post->ID, 'wordchat_synced', true);
        $synced_at = get_post_meta($post->ID, 'wordchat_synced_at', true);
        $published = get_the_date('Y-m-d H:i', $post);
        $modified = get_the_modified_date('Y-m-d H:i', $post);
        $needs_resync = $synced_at && strtotime($modified) > strtotime($synced_at);

        $status = '⬜ ' . esc_html__('Not Synced', 'wordchat');
        if ($is_synced) {
            $status = $needs_resync ? '⚠️ ' . esc_html__('Needs Resync', 'wordchat') : '✅ ' . esc_html__('Synced', 'wordchat');
        }

        echo '<tr>';
        echo '<td><input type="checkbox" name="wordchat_sync[]" value="' . esc_attr($post->ID) . '"' . ($is_synced ? ' checked' : '') . '></td>';
        echo '<td>' . esc_html($post->post_title) . '</td>';
        echo '<td>' . esc_html($post->post_type) . '</td>';
        echo '<td>' . esc_html($published) . '</td>';
        echo '<td>' . esc_html($modified) . '</td>';
        echo '<td>' . $status . '</td>';
        echo '<td>' . ($synced_at ? esc_html($synced_at) : '—') . '</td>';
        echo '</tr>';
    }

    echo '</tbody></table><p><input type="submit" class="button button-primary" value="' . esc_attr__('Sync Selected', 'wordchat') . '"></p></form></div>';
}

// Add to admin menu
function wordchat_add_sync_page() {
    add_submenu_page(
        'wordchat',
        __('Content Sync', 'wordchat'),
        __('Content Sync', 'wordchat'),
        'manage_options',
        'wordchat-sync',
        'wordchat_render_sync_page'
    );
}
add_action('admin_menu', 'wordchat_add_sync_page');
