#!/bin/bash
# WordChat Plugin Release Script
#
# Automates the entire release process:
# 1. Version bump
# 2. Build plugin ZIP
# 3. Commit changes
# 4. Create git tag
# 5. Push to GitHub
# 6. Create GitHub release with ZIP
# 7. Update database with new version
#
# Usage:
#   ./release.sh 0.1.28
#   ./release.sh 0.1.28 --skip-db

set -e

# Colors
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m'

# Check arguments
if [ -z "$1" ]; then
    echo -e "${RED}Error: Version number required${NC}"
    echo "Usage: ./release.sh 0.1.28"
    exit 1
fi

NEW_VERSION=$1
SKIP_DB=false

if [[ "$@" == *"--skip-db"* ]]; then
    SKIP_DB=true
fi

echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}WordChat Plugin Release: v${NEW_VERSION}${NC}"
echo -e "${BLUE}========================================${NC}"
echo ""

# Confirm
read -p "Release version ${NEW_VERSION}? (y/n): " -n 1 -r
echo
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Cancelled."
    exit 1
fi

# Step 1: Update version in files
echo -e "${YELLOW}Step 1: Updating version numbers...${NC}"

# Update plugin header
sed -i '' "s/Version: .*/Version: ${NEW_VERSION}/" wordchat.php
echo "  ✓ Updated wordchat.php header"

# Update constant
sed -i '' "s/define('WORDCHAT_VERSION', '.*');/define('WORDCHAT_VERSION', '${NEW_VERSION}');/" wordchat.php
echo "  ✓ Updated WORDCHAT_VERSION constant"

# Update readme.txt stable tag
sed -i '' "s/Stable tag: .*/Stable tag: ${NEW_VERSION}/" readme.txt
echo "  ✓ Updated readme.txt"

# Step 2: Build plugin ZIP
echo ""
echo -e "${YELLOW}Step 2: Building plugin ZIP...${NC}"
./build.sh

if [ ! -f "wordchat-pro-${NEW_VERSION}.zip" ]; then
    echo -e "${RED}Error: Build failed - ZIP not found${NC}"
    exit 1
fi

ZIP_SIZE=$(du -h "wordchat-pro-${NEW_VERSION}.zip" | cut -f1)
echo -e "${GREEN}  ✓ Built wordchat-pro-${NEW_VERSION}.zip (${ZIP_SIZE})${NC}"

# Step 3: Commit changes
echo ""
echo -e "${YELLOW}Step 3: Committing changes...${NC}"

git add wordchat.php readme.txt "wordchat-pro-${NEW_VERSION}.zip"
git commit -m "release: v${NEW_VERSION}

- Version bump to ${NEW_VERSION}
- Built plugin ZIP for distribution
- Ready for GitHub release"

echo -e "${GREEN}  ✓ Changes committed${NC}"

# Step 4: Create git tag
echo ""
echo -e "${YELLOW}Step 4: Creating git tag...${NC}"

git tag -a "v${NEW_VERSION}" -m "WordChat Pro v${NEW_VERSION}

Release notes: See GitHub release page"

echo -e "${GREEN}  ✓ Tag v${NEW_VERSION} created${NC}"

# Step 5: Push to GitHub
echo ""
echo -e "${YELLOW}Step 5: Pushing to GitHub...${NC}"

git push origin main
git push origin "v${NEW_VERSION}"

echo -e "${GREEN}  ✓ Pushed to GitHub${NC}"

# Step 6: Create GitHub release
echo ""
echo -e "${YELLOW}Step 6: Creating GitHub release...${NC}"

# Generate changelog (you can customize this)
CHANGELOG="**WordChat Pro v${NEW_VERSION}**

## Changes
- Bug fixes and improvements
- Enhanced performance and stability

## Installation
1. Download \`wordchat-pro-${NEW_VERSION}.zip\`
2. Upload to WordPress via Plugins > Add New > Upload Plugin
3. Activate and enter your license key

## Requirements
- WordPress 5.5+
- PHP 7.4+

## Update Instructions
If you have an active Pro license, you'll receive this update automatically in your WordPress dashboard.

---

**Support**: support@word.chat
**Website**: https://www.word.chat/"

# Create release with gh CLI
gh release create "v${NEW_VERSION}" \
    --title "WordChat Pro v${NEW_VERSION}" \
    --notes "${CHANGELOG}" \
    "wordchat-pro-${NEW_VERSION}.zip"

DOWNLOAD_URL="https://github.com/jeronimodeleon/word-chat-wp-plugin/releases/download/v${NEW_VERSION}/wordchat-pro-${NEW_VERSION}.zip"

echo -e "${GREEN}  ✓ GitHub release created${NC}"
echo -e "    ${BLUE}${DOWNLOAD_URL}${NC}"

# Step 7: Update database
if [ "$SKIP_DB" = false ]; then
    echo ""
    echo -e "${YELLOW}Step 7: Updating database...${NC}"

    if [ -z "$SUPABASE_DB_URL" ]; then
        echo -e "${YELLOW}  ⚠ SUPABASE_DB_URL not set${NC}"
        echo ""
        echo "Run this manually:"
        echo ""
        echo "psql \$SUPABASE_DB_URL <<EOF"
        echo "INSERT INTO plugin_versions ("
        echo "    version, download_url, changelog,"
        echo "    requires_wp_version, requires_php_version,"
        echo "    tested_wp_version, is_stable, released_at"
        echo ") VALUES ("
        echo "    '${NEW_VERSION}',"
        echo "    '${DOWNLOAD_URL}',"
        echo "    'Version ${NEW_VERSION} - Bug fixes and improvements',"
        echo "    '5.5', '7.4', '6.7', TRUE, NOW()"
        echo ") ON CONFLICT (version) DO UPDATE SET"
        echo "    download_url = EXCLUDED.download_url,"
        echo "    changelog = EXCLUDED.changelog;"
        echo "EOF"
    else
        psql "$SUPABASE_DB_URL" <<EOF
INSERT INTO plugin_versions (
    version, download_url, changelog,
    requires_wp_version, requires_php_version,
    tested_wp_version, is_stable, released_at
) VALUES (
    '${NEW_VERSION}',
    '${DOWNLOAD_URL}',
    'Version ${NEW_VERSION} - Bug fixes and improvements',
    '5.5', '7.4', '6.7', TRUE, NOW()
) ON CONFLICT (version) DO UPDATE SET
    download_url = EXCLUDED.download_url,
    changelog = EXCLUDED.changelog;
EOF

        echo -e "${GREEN}  ✓ Database updated${NC}"
    fi
else
    echo ""
    echo -e "${YELLOW}Step 7: Skipping database update (--skip-db)${NC}"
fi

# Summary
echo ""
echo -e "${BLUE}========================================${NC}"
echo -e "${BLUE}Release Complete! 🚀${NC}"
echo -e "${BLUE}========================================${NC}"
echo ""
echo -e "${GREEN}Version:${NC} ${NEW_VERSION}"
echo -e "${GREEN}Download:${NC} ${DOWNLOAD_URL}"
echo -e "${GREEN}GitHub:${NC} https://github.com/jeronimodeleon/word-chat-wp-plugin/releases/tag/v${NEW_VERSION}"
echo ""
echo -e "${YELLOW}Next Steps:${NC}"
echo "  1. Test download URL works"
echo "  2. Verify plugin updates show in WordPress"
echo "  3. Update marketing site if needed"
echo ""
echo -e "${GREEN}Done!${NC}"
